'use client'

import { useEffect, useState } from 'react'
import { motion } from 'framer-motion'
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card'
import { Button } from '@/components/ui/button'
import { Badge } from '@/components/ui/badge'
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table'
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog'
import { Textarea } from '@/components/ui/textarea'
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select'
import { useToast } from '@/hooks/use-toast'
import { 
  Settings, 
  CheckCircle, 
  XCircle, 
  Clock, 
  DollarSign, 
  Users, 
  Gamepad2,
  Eye,
  MessageSquare
} from 'lucide-react'

interface Order {
  id: string
  gameId: string
  pricingPlanId: string
  amount: number
  currency: string
  status: 'PENDING' | 'CONFIRMED' | 'REJECTED' | 'CANCELLED'
  paymentMethod: string
  paymentProof?: string
  customerEmail: string
  customerName: string
  customerPhone: string
  notes?: string
  adminNotes?: string
  confirmedAt?: string
  createdAt: string
  updatedAt: string
  game: {
    id: string
    name: string
    slug: string
  }
  pricingPlan: {
    id: string
    duration: string
    price: number
    currency: string
  }
}

export default function AdminPanel() {
  const [orders, setOrders] = useState<Order[]>([])
  const [selectedOrder, setSelectedOrder] = useState<Order | null>(null)
  const [isDetailDialogOpen, setIsDetailDialogOpen] = useState(false)
  const [isUpdating, setIsUpdating] = useState(false)
  const [adminNotes, setAdminNotes] = useState('')
  const { toast } = useToast()

  useEffect(() => {
    fetchOrders()
  }, [])

  const fetchOrders = async () => {
    try {
      const response = await fetch('/api/orders')
      if (response.ok) {
        const data = await response.json()
        setOrders(data)
      }
    } catch (error) {
      console.error('Failed to fetch orders:', error)
    }
  }

  const getDurationLabel = (duration: string) => {
    switch (duration) {
      case '1_day':
        return '1 Hari'
      case '1_week':
        return '1 Minggu'
      case '1_month':
        return '1 Bulan'
      default:
        return duration
    }
  }

  const formatPrice = (price: number) => {
    return new Intl.NumberFormat('id-ID', {
      style: 'currency',
      currency: 'IDR',
      minimumFractionDigits: 0,
    }).format(price)
  }

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleString('id-ID', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
    })
  }

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'PENDING':
        return 'bg-yellow-100 text-yellow-800'
      case 'CONFIRMED':
        return 'bg-green-100 text-green-800'
      case 'REJECTED':
        return 'bg-red-100 text-red-800'
      case 'CANCELLED':
        return 'bg-gray-100 text-gray-800'
      default:
        return 'bg-gray-100 text-gray-800'
    }
  }

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'PENDING':
        return <Clock className="h-4 w-4" />
      case 'CONFIRMED':
        return <CheckCircle className="h-4 w-4" />
      case 'REJECTED':
        return <XCircle className="h-4 w-4" />
      case 'CANCELLED':
        return <XCircle className="h-4 w-4" />
      default:
        return <Clock className="h-4 w-4" />
    }
  }

  const handleViewDetails = (order: Order) => {
    setSelectedOrder(order)
    setAdminNotes(order.adminNotes || '')
    setIsDetailDialogOpen(true)
  }

  const handleUpdateStatus = async (status: 'CONFIRMED' | 'REJECTED') => {
    if (!selectedOrder) return

    setIsUpdating(true)
    try {
      const response = await fetch(`/api/orders/${selectedOrder.id}`, {
        method: 'PATCH',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          status,
          adminNotes,
        }),
      })

      if (response.ok) {
        toast({
          title: 'Status Updated',
          description: `Order has been ${status.toLowerCase()} successfully.`,
        })
        setIsDetailDialogOpen(false)
        fetchOrders()
      } else {
        throw new Error('Failed to update order status')
      }
    } catch (error) {
      toast({
        title: 'Error',
        description: 'Failed to update order status. Please try again.',
        variant: 'destructive',
      })
    } finally {
      setIsUpdating(false)
    }
  }

  const getStats = () => {
    const pending = orders.filter(o => o.status === 'PENDING').length
    const confirmed = orders.filter(o => o.status === 'CONFIRMED').length
    const rejected = orders.filter(o => o.status === 'REJECTED').length
    const totalRevenue = orders
      .filter(o => o.status === 'CONFIRMED')
      .reduce((sum, o) => sum + o.amount, 0)

    return { pending, confirmed, rejected, totalRevenue }
  }

  const stats = getStats()

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Header */}
      <header className="bg-white border-b border-gray-200 sticky top-0 z-50">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-2">
              <Settings className="h-8 w-8 text-blue-600" />
              <h1 className="text-2xl font-bold text-gray-900">Admin Panel</h1>
            </div>
            <div className="flex items-center space-x-4">
              <Button onClick={fetchOrders} variant="outline">
                Refresh
              </Button>
              <a href="/" className="text-sm text-gray-600 hover:text-gray-900">
                Back to Shop
              </a>
            </div>
          </div>
        </div>
      </header>

      {/* Stats Section */}
      <section className="py-8 px-4">
        <div className="max-w-7xl mx-auto">
          <div className="grid grid-cols-1 md:grid-cols-4 gap-6">
            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Pending Orders</CardTitle>
                <Clock className="h-4 w-4 text-muted-foreground" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold text-yellow-600">{stats.pending}</div>
              </CardContent>
            </Card>
            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Confirmed Orders</CardTitle>
                <CheckCircle className="h-4 w-4 text-muted-foreground" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold text-green-600">{stats.confirmed}</div>
              </CardContent>
            </Card>
            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Rejected Orders</CardTitle>
                <XCircle className="h-4 w-4 text-muted-foreground" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold text-red-600">{stats.rejected}</div>
              </CardContent>
            </Card>
            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Total Revenue</CardTitle>
                <DollarSign className="h-4 w-4 text-muted-foreground" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold text-blue-600">
                  {formatPrice(stats.totalRevenue)}
                </div>
              </CardContent>
            </Card>
          </div>
        </div>
      </section>

      {/* Orders Table */}
      <section className="py-8 px-4">
        <div className="max-w-7xl mx-auto">
          <Card>
            <CardHeader>
              <CardTitle>Orders Management</CardTitle>
              <CardDescription>
                Manage and track all customer orders
              </CardDescription>
            </CardHeader>
            <CardContent>
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Order ID</TableHead>
                    <TableHead>Customer</TableHead>
                    <TableHead>Game</TableHead>
                    <TableHead>Duration</TableHead>
                    <TableHead>Amount</TableHead>
                    <TableHead>Status</TableHead>
                    <TableHead>Date</TableHead>
                    <TableHead>Actions</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {orders.map((order) => (
                    <TableRow key={order.id}>
                      <TableCell className="font-mono text-sm">
                        {order.id.slice(0, 8)}...
                      </TableCell>
                      <TableCell>
                        <div>
                          <div className="font-medium">{order.customerName}</div>
                          <div className="text-sm text-gray-500">{order.customerEmail}</div>
                        </div>
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center space-x-2">
                          <Gamepad2 className="h-4 w-4" />
                          <span>{order.game.name}</span>
                        </div>
                      </TableCell>
                      <TableCell>
                        {getDurationLabel(order.pricingPlan.duration)}
                      </TableCell>
                      <TableCell className="font-medium">
                        {formatPrice(order.amount)}
                      </TableCell>
                      <TableCell>
                        <Badge className={getStatusColor(order.status)}>
                          <div className="flex items-center space-x-1">
                            {getStatusIcon(order.status)}
                            <span>{order.status}</span>
                          </div>
                        </Badge>
                      </TableCell>
                      <TableCell className="text-sm text-gray-500">
                        {formatDate(order.createdAt)}
                      </TableCell>
                      <TableCell>
                        <Button
                          size="sm"
                          variant="outline"
                          onClick={() => handleViewDetails(order)}
                        >
                          <Eye className="h-4 w-4 mr-1" />
                          View
                        </Button>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </CardContent>
          </Card>
        </div>
      </section>

      {/* Order Detail Dialog */}
      <Dialog open={isDetailDialogOpen} onOpenChange={setIsDetailDialogOpen}>
        <DialogContent className="sm:max-w-2xl">
          <DialogHeader>
            <DialogTitle>Order Details</DialogTitle>
            <DialogDescription>
              Review and manage order information
            </DialogDescription>
          </DialogHeader>
          {selectedOrder && (
            <div className="space-y-6">
              {/* Order Information */}
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <h4 className="font-semibold text-sm text-gray-500 mb-1">Order ID</h4>
                  <p className="font-mono text-sm">{selectedOrder.id}</p>
                </div>
                <div>
                  <h4 className="font-semibold text-sm text-gray-500 mb-1">Status</h4>
                  <Badge className={getStatusColor(selectedOrder.status)}>
                    <div className="flex items-center space-x-1">
                      {getStatusIcon(selectedOrder.status)}
                      <span>{selectedOrder.status}</span>
                    </div>
                  </Badge>
                </div>
                <div>
                  <h4 className="font-semibold text-sm text-gray-500 mb-1">Customer</h4>
                  <p>{selectedOrder.customerName}</p>
                </div>
                <div>
                  <h4 className="font-semibold text-sm text-gray-500 mb-1">Email</h4>
                  <p>{selectedOrder.customerEmail}</p>
                </div>
                <div>
                  <h4 className="font-semibold text-sm text-gray-500 mb-1">Phone</h4>
                  <p>{selectedOrder.customerPhone}</p>
                </div>
                <div>
                  <h4 className="font-semibold text-sm text-gray-500 mb-1">Payment Method</h4>
                  <p>{selectedOrder.paymentMethod}</p>
                </div>
              </div>

              {/* Game Information */}
              <div className="border-t pt-4">
                <h4 className="font-semibold text-sm text-gray-500 mb-2">Game Information</h4>
                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <h4 className="font-semibold text-sm text-gray-500 mb-1">Game</h4>
                    <p>{selectedOrder.game.name}</p>
                  </div>
                  <div>
                    <h4 className="font-semibold text-sm text-gray-500 mb-1">Duration</h4>
                    <p>{getDurationLabel(selectedOrder.pricingPlan.duration)}</p>
                  </div>
                  <div>
                    <h4 className="font-semibold text-sm text-gray-500 mb-1">Amount</h4>
                    <p className="font-medium">{formatPrice(selectedOrder.amount)}</p>
                  </div>
                  <div>
                    <h4 className="font-semibold text-sm text-gray-500 mb-1">Order Date</h4>
                    <p>{formatDate(selectedOrder.createdAt)}</p>
                  </div>
                </div>
              </div>

              {/* Notes */}
              <div className="border-t pt-4">
                <h4 className="font-semibold text-sm text-gray-500 mb-2">Customer Notes</h4>
                <p className="text-sm bg-gray-50 p-3 rounded">
                  {selectedOrder.notes || 'No notes provided'}
                </p>
              </div>

              {/* Admin Notes */}
              <div className="border-t pt-4">
                <h4 className="font-semibold text-sm text-gray-500 mb-2">Admin Notes</h4>
                <Textarea
                  value={adminNotes}
                  onChange={(e) => setAdminNotes(e.target.value)}
                  placeholder="Add admin notes..."
                  rows={3}
                />
              </div>

              {/* Actions */}
              {selectedOrder.status === 'PENDING' && (
                <div className="flex justify-end space-x-2 border-t pt-4">
                  <Button
                    variant="outline"
                    onClick={() => handleUpdateStatus('REJECTED')}
                    disabled={isUpdating}
                  >
                    <XCircle className="h-4 w-4 mr-2" />
                    Reject
                  </Button>
                  <Button
                    onClick={() => handleUpdateStatus('CONFIRMED')}
                    disabled={isUpdating}
                  >
                    <CheckCircle className="h-4 w-4 mr-2" />
                    Confirm
                  </Button>
                </div>
              )}
            </div>
          )}
        </DialogContent>
      </Dialog>
    </div>
  )
}