'use client'

import { useEffect, useState } from 'react'
import { motion } from 'framer-motion'
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card'
import { Button } from '@/components/ui/button'
import { Badge } from '@/components/ui/badge'
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs'
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog'
import { Input } from '@/components/ui/input'
import { Label } from '@/components/ui/label'
import { Textarea } from '@/components/ui/textarea'
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select'
import { useToast } from '@/hooks/use-toast'
import { Clock, Star, Gamepad2, Users, Crown } from 'lucide-react'

interface Game {
  id: string
  name: string
  description: string
  image: string
  slug: string
  isActive: boolean
  pricingPlans: PricingPlan[]
}

interface PricingPlan {
  id: string
  duration: string
  price: number
  currency: string
  isActive: boolean
}

interface OrderFormData {
  customerName: string
  customerEmail: string
  customerPhone: string
  notes: string
  paymentMethod: string
}

export default function Home() {
  const [games, setGames] = useState<Game[]>([])
  const [selectedGame, setSelectedGame] = useState<Game | null>(null)
  const [selectedPlan, setSelectedPlan] = useState<PricingPlan | null>(null)
  const [isOrderDialogOpen, setIsOrderDialogOpen] = useState(false)
  const [isSubmitting, setIsSubmitting] = useState(false)
  const { toast } = useToast()

  const [formData, setFormData] = useState<OrderFormData>({
    customerName: '',
    customerEmail: '',
    customerPhone: '',
    notes: '',
    paymentMethod: '',
  })

  useEffect(() => {
    fetchGames()
  }, [])

  const fetchGames = async () => {
    try {
      const response = await fetch('/api/games')
      if (response.ok) {
        const data = await response.json()
        setGames(data)
      }
    } catch (error) {
      console.error('Failed to fetch games:', error)
    }
  }

  const getDurationLabel = (duration: string) => {
    switch (duration) {
      case '1_day':
        return '1 Hari'
      case '1_week':
        return '1 Minggu'
      case '1_month':
        return '1 Bulan'
      default:
        return duration
    }
  }

  const formatPrice = (price: number) => {
    return new Intl.NumberFormat('id-ID', {
      style: 'currency',
      currency: 'IDR',
      minimumFractionDigits: 0,
    }).format(price)
  }

  const handleOrderClick = (game: Game, plan: PricingPlan) => {
    setSelectedGame(game)
    setSelectedPlan(plan)
    setIsOrderDialogOpen(true)
  }

  const handleInputChange = (field: keyof OrderFormData, value: string) => {
    setFormData(prev => ({ ...prev, [field]: value }))
  }

  const handleSubmitOrder = async (e: React.FormEvent) => {
    e.preventDefault()
    if (!selectedGame || !selectedPlan) return

    setIsSubmitting(true)
    try {
      const response = await fetch('/api/orders', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          gameId: selectedGame.id,
          pricingPlanId: selectedPlan.id,
          amount: selectedPlan.price,
          currency: selectedPlan.currency,
          ...formData,
        }),
      })

      if (response.ok) {
        toast({
          title: 'Pesanan Berhasil!',
          description: 'Pesanan Anda telah dibuat dan menunggu konfirmasi pembayaran.',
        })
        setIsOrderDialogOpen(false)
        setFormData({
          customerName: '',
          customerEmail: '',
          customerPhone: '',
          notes: '',
          paymentMethod: '',
        })
      } else {
        throw new Error('Failed to create order')
      }
    } catch (error) {
      toast({
        title: 'Error',
        description: 'Gagal membuat pesanan. Silakan coba lagi.',
        variant: 'destructive',
      })
    } finally {
      setIsSubmitting(false)
    }
  }

  const getGameIcon = (gameName: string) => {
    switch (gameName.toLowerCase()) {
      case 'delta force':
        return <Crown className="h-6 w-6" />
      case 'pubg mobile':
        return <Gamepad2 className="h-6 w-6" />
      case 'mobile legend':
        return <Users className="h-6 w-6" />
      default:
        return <Gamepad2 className="h-6 w-6" />
    }
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100 dark:from-gray-900 dark:to-gray-800">
      {/* Header */}
      <header className="bg-white/80 backdrop-blur-sm border-b border-gray-200 sticky top-0 z-50">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-2">
              <Gamepad2 className="h-8 w-8 text-blue-600" />
              <h1 className="text-2xl font-bold text-gray-900">GameShop</h1>
            </div>
            <div className="flex items-center space-x-4">
              <Badge variant="secondary" className="bg-green-100 text-green-800">
                <Star className="h-4 w-4 mr-1" />
                Trusted Store
              </Badge>
              <a href="/admin" className="text-sm text-gray-600 hover:text-gray-900">
                Admin Panel
              </a>
            </div>
          </div>
        </div>
      </header>

      {/* Hero Section */}
      <section className="py-16 px-4">
        <div className="max-w-4xl mx-auto text-center">
          <motion.h1 
            className="text-4xl md:text-6xl font-bold text-gray-900 mb-6"
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.6 }}
          >
            Game Premium
            <span className="text-blue-600"> Murah</span>
          </motion.h1>
          <motion.p 
            className="text-xl text-gray-600 mb-8"
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.6, delay: 0.2 }}
          >
            Akses game favorit Anda dengan harga terbaik. Pilih paket 1 hari, 1 minggu, atau 1 bulan.
          </motion.p>
        </div>
      </section>

      {/* Games Section */}
      <section className="py-16 px-4">
        <div className="max-w-7xl mx-auto">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">Pilih Game Anda</h2>
            <p className="text-gray-600">Kami menyediakan akses premium untuk game-game populer</p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
            {games.map((game, index) => (
              <motion.div
                key={game.id}
                initial={{ opacity: 0, y: 20 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ duration: 0.6, delay: index * 0.1 }}
                whileHover={{ y: -5 }}
              >
                <Card className="h-full bg-white border-gray-200 hover:shadow-xl transition-all duration-300 transform hover:scale-105">
                  <CardHeader className="text-center">
                    <div className="flex justify-center mb-4">
                      <motion.div 
                        className="p-3 bg-blue-100 rounded-full"
                        whileHover={{ scale: 1.1, rotate: 5 }}
                        transition={{ type: "spring", stiffness: 300 }}
                      >
                        {getGameIcon(game.name)}
                      </motion.div>
                    </div>
                    <CardTitle className="text-xl font-bold text-gray-900">
                      {game.name}
                    </CardTitle>
                    <CardDescription className="text-gray-600">
                      {game.description}
                    </CardDescription>
                  </CardHeader>
                  <CardContent>
                    <div className="space-y-4">
                      {game.pricingPlans
                        .filter(plan => plan.isActive)
                        .map((plan) => (
                          <motion.div
                            key={plan.id}
                            className="flex items-center justify-between p-3 bg-gray-50 rounded-lg hover:bg-blue-50 transition-colors cursor-pointer"
                            whileHover={{ scale: 1.02 }}
                            whileTap={{ scale: 0.98 }}
                          >
                            <div className="flex items-center space-x-2">
                              <Clock className="h-4 w-4 text-gray-500" />
                              <span className="font-medium text-gray-700">
                                {getDurationLabel(plan.duration)}
                              </span>
                            </div>
                            <div className="flex items-center space-x-2">
                              <motion.span 
                                className="font-bold text-blue-600"
                                whileHover={{ scale: 1.1 }}
                              >
                                {formatPrice(plan.price)}
                              </motion.span>
                              <motion.div
                                whileHover={{ scale: 1.05 }}
                                whileTap={{ scale: 0.95 }}
                              >
                                <Button
                                  size="sm"
                                  onClick={() => handleOrderClick(game, plan)}
                                  className="bg-blue-600 hover:bg-blue-700 text-white shadow-md hover:shadow-lg transition-shadow"
                                >
                                  Pesan
                                </Button>
                              </motion.div>
                            </div>
                          </motion.div>
                        ))}
                    </div>
                  </CardContent>
                </Card>
              </motion.div>
            ))}
          </div>
        </div>
      </section>

      {/* Order Dialog */}
      <Dialog open={isOrderDialogOpen} onOpenChange={setIsOrderDialogOpen}>
        <DialogContent className="sm:max-w-md">
          <DialogHeader>
            <DialogTitle>Buat Pesanan</DialogTitle>
            <DialogDescription>
              {selectedGame && selectedPlan && (
                <>
                  {selectedGame.name} - {getDurationLabel(selectedPlan.duration)} - {formatPrice(selectedPlan.price)}
                </>
              )}
            </DialogDescription>
          </DialogHeader>
          <motion.form 
            onSubmit={handleSubmitOrder} 
            className="space-y-4"
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.3 }}
          >
            <motion.div 
              className="space-y-2"
              initial={{ opacity: 0, x: -20 }}
              animate={{ opacity: 1, x: 0 }}
              transition={{ duration: 0.3, delay: 0.1 }}
            >
              <Label htmlFor="name">Nama Lengkap</Label>
              <Input
                id="name"
                value={formData.customerName}
                onChange={(e) => handleInputChange('customerName', e.target.value)}
                required
                className="transition-all duration-200 focus:ring-2 focus:ring-blue-500"
              />
            </motion.div>
            <motion.div 
              className="space-y-2"
              initial={{ opacity: 0, x: -20 }}
              animate={{ opacity: 1, x: 0 }}
              transition={{ duration: 0.3, delay: 0.2 }}
            >
              <Label htmlFor="email">Email</Label>
              <Input
                id="email"
                type="email"
                value={formData.customerEmail}
                onChange={(e) => handleInputChange('customerEmail', e.target.value)}
                required
                className="transition-all duration-200 focus:ring-2 focus:ring-blue-500"
              />
            </motion.div>
            <motion.div 
              className="space-y-2"
              initial={{ opacity: 0, x: -20 }}
              animate={{ opacity: 1, x: 0 }}
              transition={{ duration: 0.3, delay: 0.3 }}
            >
              <Label htmlFor="phone">Nomor WhatsApp</Label>
              <Input
                id="phone"
                value={formData.customerPhone}
                onChange={(e) => handleInputChange('customerPhone', e.target.value)}
                required
                className="transition-all duration-200 focus:ring-2 focus:ring-blue-500"
              />
            </motion.div>
            <motion.div 
              className="space-y-2"
              initial={{ opacity: 0, x: -20 }}
              animate={{ opacity: 1, x: 0 }}
              transition={{ duration: 0.3, delay: 0.4 }}
            >
              <Label htmlFor="payment">Metode Pembayaran</Label>
              <Select value={formData.paymentMethod} onValueChange={(value) => handleInputChange('paymentMethod', value)}>
                <SelectTrigger className="transition-all duration-200 focus:ring-2 focus:ring-blue-500">
                  <SelectValue placeholder="Pilih metode pembayaran" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="transfer_bank">Transfer Bank</SelectItem>
                  <SelectItem value="ewallet">E-Wallet (GoPay/OVO/Dana)</SelectItem>
                  <SelectItem value="qris">QRIS</SelectItem>
                </SelectContent>
              </Select>
            </motion.div>
            <motion.div 
              className="space-y-2"
              initial={{ opacity: 0, x: -20 }}
              animate={{ opacity: 1, x: 0 }}
              transition={{ duration: 0.3, delay: 0.5 }}
            >
              <Label htmlFor="notes">Catatan (Opsional)</Label>
              <Textarea
                id="notes"
                value={formData.notes}
                onChange={(e) => handleInputChange('notes', e.target.value)}
                placeholder="Masukkan catatan tambahan jika ada"
                className="transition-all duration-200 focus:ring-2 focus:ring-blue-500"
              />
            </motion.div>
            <motion.div 
              className="flex justify-end space-x-2"
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.3, delay: 0.6 }}
            >
              <Button 
                type="button" 
                variant="outline" 
                onClick={() => setIsOrderDialogOpen(false)}
                className="transition-all duration-200 hover:scale-105"
              >
                Batal
              </Button>
              <Button 
                type="submit" 
                disabled={isSubmitting}
                className="transition-all duration-200 hover:scale-105 disabled:opacity-50"
              >
                {isSubmitting ? 'Memproses...' : 'Buat Pesanan'}
              </Button>
            </motion.div>
          </motion.form>
        </DialogContent>
      </Dialog>
    </div>
  )
}